/*
 * Copyright (C) 2004 Red Hat, Inc.
 * Copyright (C) 2006 Alberto Ruiz
 * Copyright (C) 2016 Collabora Ltd.
 *
 * SPDX-License-Identifier: LGPL-2+
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

#include "glib-derived.h"

#include <string.h>

/**
 * _trp_utf8_make_valid:
 * @str: string to coerce into UTF-8
 *
 * If the provided string is valid UTF-8, return a copy of it. If not,
 * return a copy in which bytes that could not be interpreted as valid Unicode
 * are replaced with the Unicode replacement character (U+FFFD).
 *
 * For example, this is an appropriate function to use if you have received
 * a string that was incorrectly declared to be UTF-8, and you need a valid
 * UTF-8 version of it that can be logged or displayed to the user, with the
 * assumption that it is close enough to ASCII or UTF-8 to be mostly
 * readable as-is.
 *
 * Returns: (transfer full): a valid UTF-8 string whose content resembles @str
 *
 * Since: 2.52
 */
/* FIXME: replace with g_utf8_make_valid() when we get GLib 2.51 */
gchar *
_trp_utf8_make_valid (const gchar *str)
{
  GString *string;
  const gchar *remainder, *invalid;
  gssize remaining_bytes, valid_bytes;

  g_return_val_if_fail (str != NULL, NULL);

  string = NULL;
  remainder = str;
  remaining_bytes = (gssize) strlen (str);

  while (remaining_bytes != 0)
    {
      if (g_utf8_validate (remainder, remaining_bytes, &invalid))
        break;
      valid_bytes = (gssize)(invalid - remainder);

      if (string == NULL)
        string = g_string_sized_new ((gsize)remaining_bytes);

      g_string_append_len (string, remainder, valid_bytes);
      /* append U+FFFD REPLACEMENT CHARACTER */
      g_string_append (string, "\357\277\275");

      remaining_bytes -= valid_bytes + 1;
      remainder = invalid + 1;
    }

  if (string == NULL)
    return g_strdup (str);

  g_string_append (string, remainder);

  g_assert (g_utf8_validate (string->str, -1, NULL));

  return g_string_free (string, FALSE);
}

